"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StorageService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("@nestjs/typeorm");
const typeorm_2 = require("typeorm");
const client_s3_1 = require("@aws-sdk/client-s3");
const s3_request_presigner_1 = require("@aws-sdk/s3-request-presigner");
const lib_storage_1 = require("@aws-sdk/lib-storage");
const utility_service_1 = require("../../services/utility.service");
const authentication_enum_1 = require("../../modules/authentication/enums/authentication.enum");
const storage_entity_1 = require("./storage.entity");
const environment_service_1 = require("../../services/environment.service");
const fs = require("fs");
let StorageService = class StorageService {
    constructor(repo) {
        this.repo = repo;
        this.bucket = environment_service_1.evt.R2_BUCKET_NAME;
        this.url = environment_service_1.evt.R2_ACCOUNT_ID;
        this.expiresIn = 3600;
        this.linksCache = utility_service_1.UtilityClass.cacheGenerator({
            limit: 10,
            name: `storage-links`,
        });
        this.userAccessCache = utility_service_1.UtilityClass.cacheGenerator({
            limit: 10,
            name: `storage-links-user-access`,
        });
        try {
            this.s3Client = new client_s3_1.S3Client({
                region: 'auto',
                endpoint: `https://${environment_service_1.evt.CLOUDFLARE_ACCOUNT_ID}.r2.cloudflarestorage.com`,
                credentials: {
                    accessKeyId: environment_service_1.evt.R2_ACCESS_KEY_ID,
                    secretAccessKey: environment_service_1.evt.R2_SECRET_ACCESS_KEY,
                },
            });
            this.ensureBucketExists().catch((err) => {
                console.error('Failed to ensure bucket exists:', err);
            });
        }
        catch (error) {
            console.error('s3Client error', error);
        }
    }
    async ensureBucketExists() {
        try {
            await this.s3Client.send(new client_s3_1.HeadBucketCommand({ Bucket: this.bucket }));
            console.log(`Bucket ${this.bucket} exists`);
        }
        catch (error) {
            if (error instanceof client_s3_1.NoSuchBucket || error.name === 'NotFound') {
                console.log(`Bucket ${this.bucket} does not exist, creating...`);
                try {
                    await this.s3Client.send(new client_s3_1.CreateBucketCommand({ Bucket: this.bucket }));
                    console.log(`Bucket ${this.bucket} created successfully`);
                }
                catch (createError) {
                    console.error(`Failed to create bucket ${this.bucket}:`, createError);
                    throw createError;
                }
            }
            else {
                console.error(`Error checking bucket ${this.bucket}:`, error);
                throw error;
            }
        }
    }
    async uploadFile(file, meta, config) {
        if (!file)
            utility_service_1.UtilityClass.throwError({ error: 'File is required', statusCode: 400 });
        if (meta.id) {
            await this.getFileMetadata({ id: meta.id }, config.auth);
        }
        const folder = meta.refCat;
        const suffixPath = `${meta.path ? `${meta.path}/` : ''}${meta.fileName ?? meta.refNo}`;
        const key = folder ? `${folder}/${suffixPath}` : suffixPath;
        const fileSource = file.buffer || (file.path ? fs.createReadStream(file.path) : null);
        if (!fileSource) {
            utility_service_1.UtilityClass.throwError({
                error: 'File has no buffer or path specified',
                statusCode: 400,
            });
        }
        const upload = new lib_storage_1.Upload({
            client: this.s3Client,
            params: {
                Bucket: this.bucket,
                Key: key,
                Body: fileSource,
                ContentType: file.mimetype,
            },
        });
        await upload.done();
        const fileMetadata = this.repo.create({
            ...meta,
            mimeType: file.mimetype,
            size: file.size,
            key,
            originalName: meta.originalName || file.originalname,
        });
        if (meta.id) {
            fileMetadata.updaterId = config.auth.id;
        }
        else {
            fileMetadata.orgID =
                config.auth.userType == authentication_enum_1.EAuthType.provider
                    ? config.auth.providerId
                    : meta.providerId;
            fileMetadata.creatorId = config.auth.id;
        }
        await (config.entityManager
            ? config.entityManager.delete(storage_entity_1.StorageMetadataEntity, { key: key })
            : this.repo.delete({ key: key }));
        const savedMetadata = await (config.entityManager
            ? config.entityManager.save(storage_entity_1.StorageMetadataEntity, fileMetadata)
            : this.repo.save(fileMetadata));
        const url = await this.generatePresignedUrl({ id: savedMetadata.id });
        return { key, url, metadata: savedMetadata };
    }
    async deleteFile(query, auth) {
        const sQuery = query;
        if (auth.userType == authentication_enum_1.EAuthType.provider) {
            sQuery.orgID = auth.providerId;
        }
        else if (auth.userType != authentication_enum_1.EAuthType.admin)
            sQuery.creatorId = auth.id;
        const meta = await this.repo.findOne({
            select: { key: true, id: true },
            where: sQuery,
        });
        if (!meta)
            return;
        const command = new client_s3_1.DeleteObjectCommand({
            Bucket: this.bucket,
            Key: meta.key,
        });
        await this.s3Client.send(command);
        await this.repo.delete({ id: meta.id });
    }
    async checkAccessToFile(file, auth) {
        const key = `${auth.id}_${file.id}`;
        let hasAccess = this.userAccessCache.get(key);
        if (hasAccess == null) {
            const userType = auth.userType;
            hasAccess =
                file.isPublic ||
                    userType == authentication_enum_1.EAuthType.admin ||
                    (auth.providerId
                        ? file.orgID == auth.providerId
                        : file.creatorId == auth.id);
            if (hasAccess)
                this.userAccessCache.set(key, true);
        }
        if (!hasAccess) {
            this.userAccessCache.set(key, false);
            utility_service_1.UtilityClass.throwError({ statusCode: 404 });
        }
    }
    async updateFileMetadata(id, update, config) {
        await this.checkAccessToFile({ id, ...update }, config.auth);
        const fileMetadata = [{ id }, this.repo.create(update)];
        await (config.entityManager
            ? config.entityManager.update(storage_entity_1.StorageMetadataEntity, ...fileMetadata)
            : this.repo.update(...fileMetadata));
        return this.getFileMetadata({ id });
    }
    async generatePresignedUrl(query, auth) {
        if (query.id) {
            const cacheItem = this.linksCache.get(query.id);
            if (cacheItem &&
                (!cacheItem.expiresIn || cacheItem.expiresIn > Date.now())) {
                if (auth)
                    await this.checkAccessToFile({ id: query.id, ...cacheItem }, auth);
                return cacheItem.link;
            }
        }
        const meta = await this.repo.findOne({
            select: {
                key: true,
                id: true,
                isPublic: true,
                orgID: true,
                creatorId: true,
            },
            where: query,
        });
        if (!meta)
            return null;
        if (auth)
            await this.checkAccessToFile({ id: meta.id, ...meta }, auth);
        const command = new client_s3_1.GetObjectCommand({
            Bucket: this.bucket,
            Key: meta.key,
        });
        const now = Date.now();
        const link = await (0, s3_request_presigner_1.getSignedUrl)(this.s3Client, command, {
            expiresIn: this.expiresIn,
        });
        this.linksCache.set(meta.id, {
            link,
            expiresIn: this.expiresIn ? now + this.expiresIn : null,
            ...meta,
            orgID: meta.orgID,
            creatorId: meta.orgID,
        });
        return link;
    }
    convertToResolution(query) {
        return `${query.url}/transform/resize=width:${query.width},height:${query.height},fit:crop/compress=quality:${query.quality || 90}`;
    }
    async getFileMetadata(query, auth) {
        const meta = await this.repo.findOne({ where: query });
        if (auth)
            await this.checkAccessToFile({ id: meta.id, ...meta }, auth);
        return { ...meta, url: await this.generatePresignedUrl(meta) };
    }
    async searchFileMetadata({ originalName, description, orgID, ...query }, auth) {
        if (!query.isPublic && auth.userType == authentication_enum_1.EAuthType.provider)
            orgID = auth.providerId;
        return utility_service_1.UtilityClass.search(this.repo, query, {
            baseWhere: {
                originalName: utility_service_1.UtilityClass.likeFormatter(originalName),
                description: utility_service_1.UtilityClass.likeFormatter(description),
                orgID,
            },
        });
    }
    async getFileFromCloudflare(query) {
        try {
            const metadata = await this.repo.findOne({
                where: { key: query.key || null, id: query.id || null },
            });
            if (!metadata) {
                utility_service_1.UtilityClass.throwError({
                    message: `No file found with query: ${JSON.stringify(query)}`,
                });
            }
            if (!metadata.key) {
                utility_service_1.UtilityClass.throwError({
                    message: `File found ${metadata.id} with query: ${JSON.stringify(query)}, has no key`,
                });
            }
            const command = new client_s3_1.GetObjectCommand({
                Bucket: this.bucket,
                Key: metadata.key,
            });
            const response = await this.s3Client.send(command);
            if (!response.Body) {
                throw new Error('File body is empty');
            }
            const chunks = [];
            for await (const chunk of response.Body) {
                chunks.push(chunk);
            }
            const buffer = Buffer.concat(chunks);
            console.log(`Successfully retrieved file from Cloudflare R2, size: ${buffer.length} bytes`);
            return buffer;
        }
        catch (error) {
            console.error('Error retrieving file from Cloudflare R2:', error);
            throw new Error(`Failed to retrieve file from Cloudflare R2: ${error.message}`);
        }
    }
    async saveFileToFileSystem(fileBuffer, filePath) {
        try {
            const directory = filePath.substring(0, filePath.lastIndexOf('/'));
            await fs.promises.mkdir(directory, { recursive: true });
            await fs.promises.writeFile(filePath, fileBuffer);
            console.log(`File successfully saved to ${filePath}`);
        }
        catch (error) {
            console.error('Error saving file to file system:', error);
            throw new Error(`Failed to save file to file system: ${error.message}`);
        }
    }
};
exports.StorageService = StorageService;
StorageService.path = 'storage';
exports.StorageService = StorageService = __decorate([
    (0, common_1.Injectable)(),
    __param(0, (0, typeorm_1.InjectRepository)(storage_entity_1.StorageMetadataEntity)),
    __metadata("design:paramtypes", [typeorm_2.Repository])
], StorageService);
//# sourceMappingURL=storage.service.js.map