"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseService = exports.SDKBaseService = void 0;
const typeorm_1 = require("typeorm");
const search_service_1 = require("./search.service");
const base_enum_1 = require("../enums/base.enum");
const analytics_enum_1 = require("../enums/analytics.enum");
const utility_service_1 = require("./utility.service");
const shared_repository_service_1 = require("./shared-repository.service");
const route_path_class_1 = require("../classes/route-path.class");
class SDKBaseService {
    get queryStruct() {
        return this._queryStruct;
    }
    set queryStruct(value) {
        this._queryStruct = value;
    }
    static get tag() {
        return new route_path_class_1.RoutePath(this.path)?.tag;
    }
    constructor(repo, tableName) {
        this.repo = repo;
        this.tableName = tableName;
        this.sharedRepositoryService = new shared_repository_service_1.SharedRepositoryService();
        this._queryStruct = [];
        this.deletionStyle = base_enum_1.EDeletionType.soft;
        //#endregion
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        this.formatSearchQuery = async (query, auth) => query;
        this.formatSearchResults = async (results, 
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        auth) => results;
        // this.deletionStyle = null;
    }
    /**
     * @param data ```true``` indicates that the action should be blocked from the controller
     */
    updateControllerBlock(data) {
        if (!this._blockController)
            this._blockController = {};
        for (const _key in base_enum_1.EControllerBlock) {
            const key = base_enum_1.EControllerBlock[_key];
            if (Object.prototype.hasOwnProperty.call(data, key))
                this._blockController[key] = data[key];
        }
    }
    get blockController() {
        return this._blockController ? { ...this._blockController } : null;
    }
    get isHardDeletion() {
        return this.deletionStyle == base_enum_1.EDeletionType.hard;
    }
    async getById(id, auth) {
        return this.getSingle({ id }, auth);
    }
    async getSingle(where, auth) {
        auth;
        return (await this.repo.findOne({
            where: this.whereWrapper(where),
        }));
    }
    async getWhere(where) {
        return (await this.repo.find({
            where: this.whereWrapper(where),
        }));
    }
    async checkIfExistsById(id, config) {
        return this.sharedRepositoryService.checkIfExistsById(this.repo, id, config);
    }
    async checkIfExistsBy(where, config) {
        return this.sharedRepositoryService.checkIfExistsBy(this.repo, where, config);
    }
    async checkIfUniqueById(id, config) {
        return this.sharedRepositoryService.checkIfUniqueById(this.repo, id, config);
    }
    async checkIfUniqueBy(where, config) {
        return this.sharedRepositoryService.checkIfUniqueBy(this.repo, where, config);
    }
    async checkExistenceBy(where, config) {
        return this.sharedRepositoryService.checkExistenceBy(this.repo, where, config);
    }
    async checkIfDeletable(param1, config) {
        const errorMessage = config?.errorMessage ||
            `This item cannot be deleted due to its relationship with ${String(config?.relationField)}`;
        if (typeof param1 == 'function') {
            if ((await param1()) == true)
                return true;
        }
        else {
            const saved = (await this.repo.findOne({
                where: param1,
                select: { id: true, [config?.relationField]: { id: true } },
                relations: { [config?.relationField]: true },
            }));
            if (!saved)
                return true;
            const relation = saved[config?.relationField];
            if (!relation || (Array.isArray(relation) && relation.length == 0))
                return true;
        }
        utility_service_1.UtilityClass.throwError({
            statusCode: 400,
            message: errorMessage,
        });
    }
    whereWrapper(where) {
        return where;
    }
    async find(where) {
        const item = await this.repo.findBy(this.whereWrapper(where));
        return item;
    }
    async getAllIDs(where) {
        return this.sharedRepositoryService.getAllIDs(this.repo, where);
    }
    async findOne(data) {
        return (await this.find(data))?.[0];
    }
    //#region UPDATE
    async _updateChecks(id, data) {
        data;
        return;
    }
    /**
     * Set the implementation of the pre update step. It is invoked immediately before update.
     * @returns
     */
    async _preUpdateFunction(id, data) {
        id;
        data;
    }
    /**
     * Set the implementation of the post update step. It is invoked after update.
     * @returns
     */
    async _postUpdateFunction(id, data) {
        id;
        data;
    }
    /**
     * Set the implementation of the update step only.
     * @returns
     */
    async _updateFunction(id, data, config) {
        await (config?.entityManager
            ? config?.entityManager.save(this.repo.create({ ...data, id }))
            : this.repo.update({ id }, this.repo.create(data)));
    }
    async _updateByID(id, data, config) {
        if (data?.updater?.id == null)
            delete data?.updater?.id;
        if (data?.updaterId == null)
            delete data?.updaterId;
        await this.checkIfExistsById(id, { throwError: true });
        await this._updateChecks?.(id, data);
        await this._preUpdateFunction(id, data);
        await this._updateFunction(id, data, config);
        await this._postUpdateFunction(id, data);
        return config?.returnItem === false ? null : this.getById(id);
    }
    //#endregion
    //#region DELETION
    /**
     * Set the implementation of the pre deletion step. It is invoked before deletion.
     * @param ids array of id strings
     * @returns
     */
    async _preDeletionFunction(ids) {
        ids;
    }
    /**
     * Set the implementation of the post deletion step. It is invoked after deletion.
     * @param ids array of id strings
     * @returns
     */
    async _postDeletionFunction(ids) {
        ids;
    }
    /**
     * Set the implementation of only the deletion step. Checks should be handled elsewhere
     * @param ids array of id strings
     * @returns
     */
    async _deletionFunction(ids) {
        return await (this.isHardDeletion
            ? this.repo.delete({ id: (0, typeorm_1.In)(ids) })
            : this.repo.softDelete({ id: (0, typeorm_1.In)(ids) }));
    }
    /**
     * Delete a record by id. It contains checks and pre/post steps also.
     * @param id
     * @returns
     */
    async _deleteByID(id, auth) {
        return this._deleteByIDs([id], auth);
    }
    /**
     * Delete multiple records by ids. It contains checks and pre/post steps also.
     * @param ids
     * @returns
     */
    async _deleteByIDs(ids, auth) {
        await this._deleteChecks?.(ids, auth);
        await this._preDeletionFunction(ids);
        const res = this._deletionFunction(ids);
        await this._postDeletionFunction(ids);
        return res;
    }
    /**
     * Delete multiple records. It contains checks and pre/post steps also.
     * @returns
     */
    async _deleteBy(where, auth) {
        const ids = await this.getAllIDs(this.whereWrapper(where));
        return this._deleteByIDs(ids, auth);
    }
    async _deleteChecks(ids, auth) {
        ids;
        auth;
        return;
    }
    //#endregion
    //#region RESTORATION
    /**
     * Set the implementation of only the restoration step. Checks should be handled elsewhere
     * @param ids array of id strings
     * @returns
     */
    async _deletionRestorationFunction(ids) {
        return await this.repo.restore({ id: (0, typeorm_1.In)(ids) });
    }
    /**
     * Set the implementation of the pre restoration step. It is invoked before restoration
     * @param ids array of id strings
     * @returns
     */
    async _preRestorationFunction(ids) {
        ids;
    }
    /**
     * Set the implementation of the post restoration step. It is invoked after restoration
     * @param ids array of id strings
     * @returns
     */
    async _postRestorationFunction(ids) {
        ids;
    }
    /**
     * Call this if your class uses a soft deletion technique. It restores a deleted record by id. It contains checks and pre/post steps also.
     * @param id
     * @returns
     */
    async _restoreByID(id) {
        return this._restoreByIDs([id]);
    }
    /**
     * Call this if your class uses a soft deletion technique. It restores deleted records by their ids. It contains checks and pre/post steps also.
     * @param ids array of id strings
     * @returns
     */
    async _restoreByIDs(ids) {
        await this._preRestorationFunction(ids);
        const res = this._deletionRestorationFunction(ids);
        await this._postRestorationFunction(ids);
        return res;
    }
    /**
     * Restore multiple records. It contains checks and pre/post steps also.
     * @returns
     */
    async _restoreBy(where) {
        const ids = await this.getAllIDs(this.whereWrapper(where));
        return this._restoreByIDs(ids);
    }
    async search(query, auth) {
        auth;
        const res = await search_service_1.SearchService.search({
            repository: this.repo,
            tableName: this.tableName,
        }, await this.formatSearchQuery(query, auth), this.queryStruct);
        res.data?.forEach((x) => {
            if (x.creator)
                x.creator = {
                    firstname: x.creator.firstname,
                    lastname: x.creator.lastname,
                    id: x.creator.id,
                };
        });
        return await this.formatSearchResults(res, auth);
    }
    async _save(data, config) {
        return await (data.id
            ? this._updateByID(data.id, data, config)
            : this._create(data, config));
    }
    // /**@deprecated use {@link SDKBaseService._deleteChecks} instead*/
    // async _deleteByIDChecks(id: string) {
    //   id;
    //   return;
    // }
    //#region CREATE
    async _createChecks(data) {
        data;
        return;
    }
    /**
     * Set the implementation of the pre create step. It is invoked immediately before creation.
     * @returns
     */
    async _preCreateFunction(data) {
        data;
    }
    /**
     * Set the implementation of the post create step. It is invoked after creation.
     * @returns
     */
    async _postCreateFunction(requestData, savedData) {
        requestData;
        savedData;
    }
    /**
     * Set the implementation of the create step only.
     * @returns
     */
    async _createFunction(requestData, config) {
        return (await (config?.entityManager
            ? config.entityManager.save(this.repo.create(requestData))
            : this.repo.save(this.repo.create(requestData))));
    }
    async _create(data, config) {
        delete data.id;
        await this._createChecks?.(data);
        if (data?.creator?.id == null)
            delete data?.creator?.id;
        if (data?.creatorId == null)
            delete data?.creatorId;
        await this._preCreateFunction(data);
        const entity = await this._createFunction(data, config);
        await this._postCreateFunction(data, entity);
        return config?.returnItem === false
            ? null
            : config?.entityManager
                ? entity
                : this.getById(entity.id);
    }
    //#endregion
    async count(where) {
        return this.repo.count({ where: this.whereWrapper(where) });
    }
    async getAnalytics(params) {
        if (params?.id)
            return [];
        else {
            const currentYr = new Date().getFullYear();
            return Promise.all([
                {
                    header: 'Last 12 Months Creation',
                    type: analytics_enum_1.EChartType.bar,
                    data: await this.countExtractor(utility_service_1.UtilityClass.monthBackOrdering[new Date().getMonth()].map((m) => ({
                        label: m.short,
                        where: {
                            createdAt: (0, typeorm_1.Like)(`${currentYr}-${m.isoStr}-%`),
                        },
                    }))),
                },
                {
                    header: 'Active Status',
                    type: analytics_enum_1.EChartType.pie,
                    data: await this.countExtractor([
                        { label: 'Active', where: { active: true } },
                        { label: 'Inactive', where: { active: false } },
                    ]),
                },
            ]);
        }
    }
    async countExtractor(queries, repository) {
        const repo = repository || this.repo;
        if (Array.isArray(queries))
            return Promise.all(queries.map(async (x) => ({
                label: x.label,
                value: await repo.count({ where: this.whereWrapper(x.where) }),
            })));
        else {
            const columnName = queries.toString();
            const distinctValues = (await repo
                .createQueryBuilder(this.tableName)
                .select(`${columnName}`)
                .distinct(true)
                .andWhere(`${this.tableName}.active = 1`)
                .getRawMany()).map((x) => x[columnName]);
            return Promise.all(distinctValues.map(async (x) => ({
                label: x,
                value: await repo.count({ where: { [columnName]: x } }),
            })));
        }
    }
    sumExtractor(queries) {
        return Promise.all(queries.map(async (x) => ({
            label: x.label,
            value: await this.repo.sum(x.column, this.whereWrapper(x.where)),
        })));
    }
    async toggleActive(id, body) {
        if (!(await this.checkIfExistsById(id)))
            utility_service_1.UtilityClass.throwError({
                message: `ID does not exist`,
                statusCode: 404,
            });
        await this.repo.update({ id }, body);
        return body;
    }
    async toggleBulkActive({ ids, ...body }) {
        const res = await this.repo.update(this.whereWrapper({ id: (0, typeorm_1.In)(ids) }), body);
        return { rows: res.affected };
    }
    async toggleAllActive(body) {
        const res = await this.repo.update(this.whereWrapper({}), body);
        return { rows: res.affected };
    }
    async arrangeItems(body) {
        await Promise.all(body.items.map((i) => this.repo.update(this.whereWrapper({ id: i.id }), { order: i.order })));
        return true;
    }
}
exports.SDKBaseService = SDKBaseService;
class BaseService extends SDKBaseService {
    constructor(repo, tableName) {
        super(repo, tableName);
        this.repo = repo;
        this.tableName = tableName;
    }
}
exports.BaseService = BaseService;
//# sourceMappingURL=base.service.js.map