"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SearchService = void 0;
const search_enum_1 = require("../enums/search.enum");
const environment_service_1 = require("./environment.service");
class SearchService {
    static async search({ repository, tableName, }, query, queryStruct) {
        const queryBuild = repository.createQueryBuilder(tableName);
        const qs = (queryStruct || []).concat(this.defaultQueryStruct);
        for (const queryParam of qs) {
            let _tableName = tableName;
            const prefix = `${_tableName}.${queryParam.field?.toString()}`;
            if (queryParam.condition == search_enum_1.ESearchCondition.leftJoin)
                queryBuild.leftJoinAndSelect(prefix, queryParam.tableAlias);
            else {
                _tableName = queryParam.tableAlias || tableName;
                const joinerFunc = (query, parameters) => queryParam.joinCondition == search_enum_1.ESearchJoinCondition.or
                    ? queryBuild.orWhere(query, parameters)
                    : queryBuild.andWhere(query, parameters);
                if (queryParam.condition == search_enum_1.ESearchCondition.between) {
                    if (query[queryParam.upperRange] != null &&
                        query[queryParam.lowerRange] != null)
                        joinerFunc(`${prefix} BETWEEN '${query[queryParam.lowerRange]}' AND '${query[queryParam.upperRange]}'`);
                }
                else if (query[queryParam.field] != null) {
                    let prefixed = `${_tableName}.${(queryParam.actualField || queryParam.field)?.toString()}`;
                    if (queryParam.subFields?.length)
                        prefixed = `${_tableName}.${queryParam.subFields[0]} ${queryParam.subFields
                            .slice(1)
                            .map((x, index) => `->${index + 2 == queryParam.subFields.length ? '>' : ''} '${x?.toString()}'`)
                            .join(' ')}`;
                    if (queryParam.condition == search_enum_1.ESearchCondition.contains)
                        joinerFunc(`${prefixed} LIKE '%${query[queryParam.field]}%'`);
                    else if (queryParam.condition == search_enum_1.ESearchCondition.in) {
                        const value = query[queryParam.field];
                        joinerFunc(`${prefixed} IN (:...${queryParam.field})`, {
                            [queryParam.field]: Array.isArray(value) ? value : [value],
                        });
                    }
                    else if (queryParam.condition == search_enum_1.ESearchCondition.inArray)
                        if (queryParam.subFields?.length)
                            joinerFunc(`(${prefixed})::jsonb ? '${query[queryParam.field]}'`);
                        else
                            joinerFunc(`'${query[queryParam.field]}'=ANY(${prefixed})`);
                    else
                        joinerFunc(`${prefixed} ${queryParam.condition || '='} '${this.formatValue(query[queryParam.field], queryParam)}'`);
                }
            }
        }
        // if (config?.isUsingSoftDeletion)
        // queryBuild.andWhere(`${tableName}._deleted != 1`);
        query.sortField = query.sortField || 'createdAt';
        query.sortDirection =
            query.sortDirection?.toUpperCase() || search_enum_1.ESortOrder.desc;
        const pageSize = +query.pageSize;
        const pageNumber = +query.pageNumber;
        queryBuild.orderBy(`${tableName}.${String(query.sortField)}`, query.sortDirection);
        if (pageNumber && pageSize)
            queryBuild.skip(pageSize * (pageNumber - 1)).take(pageSize);
        const result = await queryBuild.getManyAndCount();
        const res = {
            pageNumber,
            pageSize,
            sortDirection: query.sortDirection,
            sortField: query.sortField,
            total: result[1],
            data: result[0],
            dataLength: result[0]?.length || 0,
        };
        if (environment_service_1.evt.ENVIRONMENT == 'dev')
            res['query'] = queryBuild.getQuery();
        return res;
    }
    static formatValue(value, queryParam) {
        if (queryParam.type === 'boolean')
            return value == 'true' || value == true ? 1 : 0;
        else if (typeof value === 'boolean')
            return value ? 1 : 0;
        else
            return value == null ? queryParam.value : value;
    }
}
exports.SearchService = SearchService;
SearchService.refQueryStruct = [
    { field: 'refCat' },
    { field: 'refNo' },
    { field: 'slug' },
];
SearchService.searchStructToObject = (struct) => {
    const obj = {};
    for (const item of struct) {
        if (item.lowerRange) {
            obj[item.lowerRange] = null;
            obj[item.upperRange] = null;
        }
        else
            obj[item.field] = null;
    }
    return obj;
};
SearchService.defaultQueryStruct = [
    {
        condition: search_enum_1.ESearchCondition.between,
        lowerRange: 'createdFrom',
        upperRange: 'createdTo',
        field: 'createdAt',
    },
    {
        condition: search_enum_1.ESearchCondition.between,
        lowerRange: 'updatedFrom',
        upperRange: 'updatedTo',
        field: 'updatedAt',
    },
    {
        condition: search_enum_1.ESearchCondition.equal,
        field: 'createdAt',
    },
    {
        condition: search_enum_1.ESearchCondition.equal,
        field: 'updatedAt',
    },
    {
        condition: search_enum_1.ESearchCondition.equal,
        field: 'creatorId',
    },
    {
        condition: search_enum_1.ESearchCondition.equal,
        field: 'active',
        type: 'boolean',
    },
    {
        condition: search_enum_1.ESearchCondition.equal,
        field: 'orgID',
    },
];
//# sourceMappingURL=search.service.js.map