"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UtilityClass = void 0;
const common_1 = require("@nestjs/common");
const logger_service_1 = require("./logger.service");
const environment_service_1 = require("./environment.service");
const swagger_1 = require("@nestjs/swagger");
const ts_helpers_1 = require("@serene-dev/ts-helpers");
const unique_names_generator_1 = require("unique-names-generator");
const os_1 = require("os");
const typeorm_1 = require("typeorm");
class UtilityClass {
    // static swaggerUILink: string = 'http://localhost:3000/api-docs';
    static backOrderingArray(arr, key) {
        const backOrderings = {}, _key = key;
        for (let index = 0; index < arr.length; index++) {
            backOrderings[arr[index][_key]] = arr
                .slice(index + 1, arr.length)
                .concat(arr.slice(0, index + 1));
        }
        // console.log(backOrderings);
        return backOrderings;
    }
    static get generateUserName() {
        return (0, unique_names_generator_1.uniqueNamesGenerator)({
            dictionaries: [unique_names_generator_1.names],
        });
    }
    static get generateStarwars() {
        return (0, unique_names_generator_1.uniqueNamesGenerator)({
            dictionaries: [unique_names_generator_1.starWars],
        });
    }
    static getOSSystem() {
        return {
            arch: os_1.default.arch(),
            availableParallelism: os_1.default.availableParallelism(),
            constants: os_1.default.constants,
            cpus: os_1.default.cpus(),
            endianness: os_1.default.endianness(),
            freemem: os_1.default.freemem(),
            getPriority: os_1.default.getPriority(),
            homedir: os_1.default.homedir(),
            hostname: os_1.default.hostname(),
            loadavg: os_1.default.loadavg(),
            machine: os_1.default.machine(),
            networkInterfaces: os_1.default.networkInterfaces(),
            platform: os_1.default.platform(),
            release: os_1.default.release(),
            tmpdir: os_1.default.tmpdir(),
            totalmem: os_1.default.totalmem(),
            type: os_1.default.type(),
            uptime: os_1.default.uptime(),
            userInfo: os_1.default.userInfo(),
            version: os_1.default.version(),
        };
    }
    static removeNull(obj) {
        const res = {};
        for (const key in obj) {
            if (obj[key] != null)
                res[key] = obj[key];
        }
        return res;
    }
    static computeArrayFieldQuery(query) {
        return Array.isArray(query) ? (0, typeorm_1.In)(query) : query;
    }
    static otpGenerator(length = 6) {
        length = length || 6;
        const randomNumber = () => Math.floor(Math.random() * 10);
        const gen = (len = 3) => new Array(len)
            .fill(true)
            .map(() => randomNumber())
            .join('');
        const starter = randomNumber(), isEven = length % 2 == 0;
        return (starter +
            gen((isEven ? length / 2 : (length - 1) / 2) - 1) +
            starter +
            gen(isEven ? length / 2 - 1 : (length - 1) / 2));
    }
    static pathToLink(file) {
        return `${environment_service_1.evt.SERVER_URL}/${file.path}/${file.fileExtension}`;
    }
    static get getHost() {
        return environment_service_1.evt.SERVER_URL;
    }
    static enableSwagger(app, config) {
        const swaggerConfig = new swagger_1.DocumentBuilder()
            .setTitle(config.appName)
            .setDescription(`The ${config.appName} API description`)
            .setVersion(config?.versionNo || '1.0')
            .addBearerAuth()
            .build();
        const document = swagger_1.SwaggerModule.createDocument(app, swaggerConfig, {
            include: config.modules,
        });
        swagger_1.SwaggerModule.setup(config.path, app, document, {
            swaggerOptions: {
                tagsSorter: 'alpha',
                operationsSorter: 'alpha',
                docExpansion: config.docExpansion,
                ...config?.swaggerOptions,
            },
        });
    }
    static likeFormatter(value) {
        return value ? (0, typeorm_1.Like)(`%${value}%`) : null;
    }
    static setDbTimezoneOffset(hours) {
        this.dbTimezoneOffset = hours * 60 * 60 * 1000;
    }
    static patchUpdater(body, auth) {
        return this.patchOwner(body, auth, 'updater');
    }
    static patchCreator(body, auth) {
        return this.patchOwner(body, auth, 'creator');
    }
    static patchOwner(body, auth, authField) {
        if (!auth?.id)
            return body;
        body[authField] = {
            id: auth.id || null,
        };
        body[authField + 'Id'] = auth.id || null;
        return body;
    }
    static get generateUUID() {
        let d = new Date().getTime();
        let d2 = (typeof performance !== 'undefined' &&
            performance.now &&
            performance.now() * 1000) ||
            0; //Time in microseconds since page-load or 0 if unsupported
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
            let r = Math.random() * 16; //random number between 0 and 16
            if (d > 0) {
                //Use timestamp until depleted
                r = (d + r) % 16 | 0;
                d = Math.floor(d / 16);
            }
            else {
                //Use microseconds since page-load if supported
                r = (d2 + r) % 16 | 0;
                d2 = Math.floor(d2 / 16);
            }
            return (c === 'x' ? r : (r & 0x3) | 0x8).toString(16);
        });
    }
    static computeRangeQuery(exactValue, lowerRange, upperRange) {
        return exactValue != null
            ? exactValue
            : upperRange && lowerRange
                ? (0, typeorm_1.Between)(lowerRange, upperRange)
                : upperRange
                    ? (0, typeorm_1.LessThanOrEqual)(upperRange)
                    : lowerRange
                        ? (0, typeorm_1.MoreThanOrEqual)(lowerRange)
                        : null;
    }
    static returnEnums(enums) {
        try {
            const res = {};
            for (const key in enums) {
                if (Object.prototype.hasOwnProperty.call(enums, key)) {
                    res[key] = Object.values(enums[key]).map((v) => ({
                        value: v,
                        label: v.replaceAll('_', ' ').replaceAll('-', ' '),
                    }));
                }
            }
            return this.handleSuccess({
                data: res,
            });
        }
        catch (error) {
            return _a.handleError(error);
        }
    }
    static returnEnum(enumm) {
        try {
            return this.handleSuccess({
                data: Object.values(enumm).map((v) => ({
                    value: v,
                    label: v.replaceAll('_', ' ').replaceAll('-', ' '),
                })),
            });
        }
        catch (error) {
            return _a.handleError(error);
        }
    }
    static async searchGeneric(repository, searchQuery, config) {
        for (const key in searchQuery)
            if (Object.prototype.hasOwnProperty.call(searchQuery, key))
                if (searchQuery[key] == 'true')
                    searchQuery[key] = 1;
                else if (searchQuery[key] == 'false')
                    searchQuery[key] = 0;
        const pageSize = +config.pagination.pageSize;
        const pageNumber = +config.pagination.pageNumber;
        const result = await repository.findAndCount({
            where: Array.isArray(config?.baseWhere)
                ? config?.baseWhere.map((w) => ({ ...w, ...searchQuery }))
                : {
                    ...config?.baseWhere,
                    ...searchQuery,
                },
            select: config?.baseSelect,
            relations: config?.baseRelations,
            order: config.pagination.sortField
                ? {
                    [config.pagination.sortField]: config.pagination.sortDirection || 'desc',
                }
                : config?.baseOrder,
            take: pageSize || null,
            skip: pageNumber && pageSize ? pageSize * (pageNumber - 1) : null,
        });
        const res = {
            pageNumber,
            pageSize,
            sortDirection: config.pagination.sortDirection,
            sortField: config.pagination.sortField,
            total: result[1],
            data: result[0],
            dataLength: result[0]?.length || 0,
        };
        return res;
    }
    static async search(repository, query, config) {
        return this.searchView(repository, query, {
            ...config,
            baseRelations: { ...config?.baseRelations, creator: true, updater: true },
        });
    }
    static async searchView(repository, { createdAt, createdFrom, createdTo, updatedAt, updatedFrom, updatedTo, ...query }, config) {
        const { pageSize: qPageSize, pageNumber: qPageNumber, sortDirection, sortField, ...searchQuery } = query || {};
        const pageSize = +qPageSize;
        const pageNumber = +qPageNumber;
        const extraWhere = {
            createdAt: config?.baseWhere?.['createdAt'] ||
                this.computeRangeQuery(createdAt, createdFrom, createdTo),
            updatedAt: config?.baseWhere?.['updatedAt'] ||
                this.computeRangeQuery(updatedAt, updatedFrom, updatedTo),
        };
        return this.searchGeneric(repository, searchQuery, {
            pagination: {
                pageSize,
                pageNumber,
                sortDirection: sortDirection?.toLowerCase(),
                sortField,
            },
            baseOrder: config?.baseOrder,
            baseSelect: config?.baseSelect,
            baseRelations: config?.baseRelations || null,
            baseWhere: Array.isArray(config?.baseWhere)
                ? config?.baseWhere.map((w) => ({ ...w, ...extraWhere }))
                : {
                    ...config?.baseWhere,
                    ...extraWhere,
                },
        });
    }
    static strConcatenator(field1, field2, joiner = ' ') {
        // debugger
        // console.log('field1',field1)
        field1 = field1?.toString()?.trim();
        field2 = field2?.toString()?.trim();
        if (field1 != null && field2 != null)
            return field1 + joiner + field2;
        else
            return field1 || field2;
    }
}
exports.UtilityClass = UtilityClass;
_a = UtilityClass;
UtilityClass.sdkUtility = new ts_helpers_1.UtilityClass({ production: environment_service_1.evt.ENVIRONMENT != 'dev' });
UtilityClass.handleSuccess = (res) => {
    return res?.data || res?.message;
    // return { data: res?.data, message: res?.message || `Success` };
};
UtilityClass.monthBackOrdering = _a.backOrderingArray(ts_helpers_1.HHConfig.Months, 'id');
UtilityClass.castObject = (newObejct) => newObejct;
/**
 * This handles user defined errors and system generated errors
 * @param res
 */
UtilityClass.handleError = (res) => {
    logger_service_1.logger.error(res?.message || res);
    throw new common_1.HttpException({
        message: res?.message || `Error occurred`,
        error: !res?.error && !res?.message ? res : res.error,
    }, res?.statusCode || common_1.HttpStatus.INTERNAL_SERVER_ERROR);
};
// Time to add to date from db ms
UtilityClass.dbTimezoneOffset = 0;
UtilityClass.columnTimestamp = {
    transformer: {
        from: (v) => {
            if (v)
                v.setTime(v.getTime() + _a.dbTimezoneOffset);
            return v;
        },
        to: (v) => v,
    },
};
UtilityClass.transformerNumber = {
    transformer: { from: (v) => +v, to: (v) => v },
};
UtilityClass.transformerBoolean = {
    transformer: { from: (v) => +v, to: (v) => v },
};
/**
 * This is a user defined error
 * @param payload
 */
UtilityClass.throwError = (payload) => {
    // debugger;
    throw {
        error: payload?.error,
        message: payload?.message || `Error occurred`,
        statusCode: payload?.statusCode || common_1.HttpStatus.BAD_REQUEST,
    };
};
UtilityClass.handleClassValidatorErrors = (e) => _a.throwError({
    error: e.flatMap((x) => Object.values(x.constraints)),
    statusCode: 400,
});
//# sourceMappingURL=utility.service.js.map