"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UtilityClass = void 0;
const tslib_1 = require("tslib");
const index_config_1 = require("../config/index.config");
const jspdf_1 = tslib_1.__importDefault(require("jspdf"));
const html2canvas_1 = tslib_1.__importDefault(require("html2canvas"));
const papaparse_1 = tslib_1.__importDefault(require("papaparse"));
const js_file_download_1 = tslib_1.__importDefault(require("js-file-download"));
const rxjs_1 = require("rxjs");
const lodash_1 = require("lodash");
const index_model_1 = require("../shared/models/index.model");
class UtilityClass {
    constructor(environment) {
        this.environment = environment;
        this.downloader = js_file_download_1.default;
        this.config = index_config_1.Config;
        this.eValidationType = index_model_1.EValidationType;
        this.ePeriod = index_model_1.EPeriod;
        this.imageTypes = {
            'image/apng': true,
            'image/avif': true,
            'image/gif': true,
            'image/jpeg': true,
            'image/png': true,
            'image/svg+xml': true,
            'image/webp': true,
        };
        /**
         * Gets all the fields in all the items in an array
         * @param arr
         * @returns The index map of the fields with the fields serving as the index keys. Each entry contains the count of the times that a value was found for the field
         */
        this.getAllFieldsInArray = (arr) => {
            const ret = {};
            for (const item of arr) {
                for (const key in item) {
                    if (Object.prototype.hasOwnProperty.call(item, key)) {
                        const element = item[key];
                        const existingItem = ret[key];
                        if (existingItem && element != null)
                            existingItem.count++;
                        ret[key] = { field: key, count: element != null ? 1 : 0 };
                    }
                }
            }
            return ret;
        };
        this.sortObjectFields = (obj) => {
            if (obj == null || typeof obj != 'object')
                return obj;
            const nObj = {};
            const fieldNames = Object.keys(obj);
            fieldNames.sort();
            // console.log(fieldNames);
            for (const key of fieldNames) {
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    let element = obj[key];
                    if (Array.isArray(element)) {
                        nObj[key] = (0, lodash_1.cloneDeep)(element);
                        for (let index = 0; index < element.length; index++) {
                            nObj[key][index] = this.sortObjectFields(element[index]);
                        }
                    }
                    else if (typeof element == 'object' && element != null) {
                        nObj[key] = this.sortObjectFields(element);
                    }
                    else
                        nObj[key] = element;
                }
            }
            return nObj;
        };
        this.xOrY = (x, y = '-') => `${x == null ? y : x}`;
        /**
         * Convert Date string to Date Time string
         * @param dateStr Date string
         * @example '2022-01-25'
         * @param config Configuration for the conversion
         * @returns Returns a Date Time string
         * @example '2022-01-25T00:00:00Z' or '2022-01-25 00:00:00'
         */
        this.dateToDateTime = (dateStr, config) => {
            const omitT = !!config?.omitT;
            if (!dateStr)
                return null;
            if (dateStr.includes('T'))
                return dateStr;
            return dateStr + (omitT ? ' 00:00:00' : 'T00:00:00Z');
        };
        /**
         * Handles the deletion of rows from an array
         * @param index Index of the row to delete
         * @param arr The array that holds the rows
         * @param deleteService The deletion service to be called (It should be an anonymous function)
         * @param addRowFunc The function (anonymous) that adds a new row to the array
         * @param cb The function to callback with the response of the deletion service as the input parameter
         */
        this.handleRowDelete = async (index, arr, deleteService, addRowFunc, cb) => {
            const id = arr[index]?.id;
            // debugger;
            if (id && deleteService) {
                const req = deleteService(id);
                if (typeof req['toPromise'] == 'function')
                    cb ? cb(await req['toPromise']()) : req['toPromise']();
                else
                    cb ? cb(await req) : null;
            }
            if (arr.length == 1) {
                arr.splice(index, 1);
                addRowFunc();
            }
            else if (arr.length > 1)
                arr.splice(index, 1);
        };
        this.isPromise = (func) => {
            return typeof func === 'object' && typeof func.then === 'function';
        };
        this.isAsync = (func) => {
            return (func.constructor.name === 'AsyncFunction' ||
                (typeof func === 'function' && this.isPromise(func())));
        };
        this.isObservable = (func) => {
            // return typeof func === 'object' && typeof func.subscribe === 'function';
            // return typeof func === 'object' && func instanceof Observable;
            // return typeof func === 'object' && func instanceof Observable<T>;
            return typeof func === 'object' && (0, rxjs_1.isObservable)(func);
            return typeof func === 'object' && typeof func.toPromise === 'function';
        };
        this.promisifyVal = (val) => {
            if (this.isAsync(val)) {
                return Promise.resolve(val);
            }
            else if (this.isObservable(val)) {
                return Promise.resolve(val.toPromise());
            }
            else {
                return Promise.resolve(val);
            }
        };
        /**
         * @param arr Array of observables to fetch
         * @returns An array of the responses merged together. It emits the merged responses as they are fetched and closes once alll the responses have been fetched
         */
        this.mergeArrObservables = (arr) => {
            return new rxjs_1.Observable((sub) => {
                const ret = [];
                let responseCount = 0;
                (0, rxjs_1.merge)(...arr.map((x) => x.$func?.pipe((0, rxjs_1.catchError)((e) => {
                    console.error(e);
                    return (0, rxjs_1.of)([]);
                }), (0, rxjs_1.tap)((r) => {
                    responseCount++;
                })))).subscribe((r) => {
                    ret.push(...r);
                    sub.next(ret);
                    if (responseCount == arr.filter((x) => x.$func).length) {
                        // debugger
                        sub.complete();
                    }
                });
            });
        };
        this.objIsEmpty = (obj, exclusionFields) => {
            if (!obj)
                return true;
            for (const key in obj) {
                if (exclusionFields?.includes(key))
                    continue;
                if (Object.prototype.hasOwnProperty.call(obj, key)) {
                    if (obj[key])
                        return false;
                }
            }
            return true;
        };
        this.dataGen = (keys, length = 10) => {
            const ret = [];
            for (let i = 0; i < length; i++) {
                let t = {};
                for (const key of keys) {
                    t[key] = this.textGen();
                }
                ret.push(t);
            }
            return ret;
        };
        this.textGen = () => 'random' + Math.round(Math.random() * 1000);
        this.trackByCustomID = (index, item) => {
            if (!item.__id) {
                item.__id = this.generateUUID();
                console.log('set custom id');
            }
            return item.__id;
        };
        this.findLabelByItem = (item, arr, labelField, keyField = 'code') => {
            if (!item)
                return null;
            const obj = arr?.find((x) => x[keyField] == item);
            if (!obj)
                return item;
            else
                return `${item} - ${obj[labelField || 'title'] || obj[labelField || 'description']}`;
        };
        this.base64toBlob = (base64Data, contentType = '', sliceSize = 512) => {
            const byteCharacters = atob(base64Data);
            const byteArrays = [];
            for (let offset = 0; offset < byteCharacters.length; offset += sliceSize) {
                const slice = byteCharacters.slice(offset, offset + sliceSize);
                const byteNumbers = new Array(slice.length);
                for (let i = 0; i < slice.length; i++) {
                    byteNumbers[i] = slice.charCodeAt(i);
                }
                const byteArray = new Uint8Array(byteNumbers);
                byteArrays.push(byteArray);
            }
            const blob = new Blob(byteArrays, { type: contentType });
            return blob;
        };
        this.base64toFile = (base64Data, fileName = Date.now() + '', contentType = '', sliceSize = 512) => {
            return new File([this.base64toBlob(base64Data, contentType, sliceSize)], fileName);
        };
        this.secondsToHourMinSec = (seconds) => {
            if (!seconds)
                return { secs: null, hours: null, mins: null };
            const hours = Math.floor(seconds / 3600) || 0, secondsExHours = seconds % 3600 || 0, mins = Math.floor(secondsExHours / 60) || 0, secs = secondsExHours % 60;
            return { hours, mins, secs };
        };
        this.minutesToDayHourMin = (minutes) => {
            if (!minutes)
                return { days: null, hours: null, mins: null };
            const days = Math.floor(minutes / 1440) || 0, miniutesExDays = minutes % 1440 || 0, hours = Math.floor(miniutesExDays / 60) || 0, mins = miniutesExDays % 60 || 0;
            return { days, hours, mins };
        };
        this.minutesToDayHourMinStr = (minutes) => {
            if (!minutes)
                return null;
            const ret = this.minutesToDayHourMin(minutes);
            return `${ret.days} ${ret.hours}:${ret.mins}`.trim();
        };
        this.minutesToDayHourMinStr2 = (minutes) => {
            if (!minutes)
                return null;
            const ret = this.minutesToDayHourMin(minutes);
            return `${ret.days}days ${ret.hours}hrs ${ret.mins}mins`.trim();
        };
        this.dayHourMinToHourMinutes = (dys, hrs, mins) => {
            if (!dys && !hrs && !mins)
                return '';
            const dysToHrs = (dys || 0) * 24;
            hrs = (+hrs || 0) + dysToHrs;
            return (hrs || '00') + ':' + (mins || '00');
        };
        this.dayHourMinToMinutes = (dys, hrs, mins) => {
            if (!dys && !hrs && !mins)
                return 0;
            const dysToHrs = (dys || 0) * 24;
            hrs = (+hrs || 0) + dysToHrs;
            return (hrs || 0) * 60 + mins;
        };
        this.hourMinToMinutes = (hrs, mins) => {
            if (!hrs && !mins)
                return null;
            return (hrs || 0) * 60 + mins;
        };
        this.hourMinToMinutes2 = (hrsMins) => {
            const [hrs, mins] = hrsMins?.split(':') || [0, 0];
            return this.hourMinToMinutes(+hrs, +mins);
        };
        this.scrollToTop = (timeout = 100) => setTimeout(() => {
            document.querySelector('.ddd')?.scrollIntoView({ behavior: 'smooth' });
            // this.uS.scrollToTop();
        }, timeout);
        this.cellStatusFormatter = (guide) => (v) => `<span class='${v == guide.successVal ? 'successCell' : v == guide.errorVal ? 'errorCell' : 'inertCell'}'>${v}</span>`;
        /**
         *
         * @returns The current date time local that can be used to set the datetime-local input
         */
        this.getLocalDateTimeNow = () => {
            const dt = new Date().toISOString().split('T');
            return dt[0] + 'T' + dt[1].split(':').slice(0, 2).join(':');
        };
        /** @returns The current date local that can be used to set the date input */
        this.localDateNow = () => {
            return new Date().toISOString().split('T')[0];
        };
        this.dateFormat = (date, format = 1) => {
            if (!date)
                return null;
            const d = new Date(date);
            if (format == 1) {
                return d.toDateString();
            }
            else if (format == 2) {
                return d.getDate() + ' ' + index_config_1.Config.Months[d.getMonth()];
            }
            else if (format == 3) {
                return d.toDateString() + ', ' + this.timeFormat(d.toLocaleTimeString());
            }
            else if (format == 4) {
                return index_config_1.Config.Months[d.getMonth()] + ' ' + d.getDate();
            }
            else {
                return '';
            }
        };
        this.toJavaDateString = (isoDateString = new Date().toISOString()) => {
            if (!isoDateString)
                return null;
            return `${isoDateString.slice(0, 19)}`;
            return `${isoDateString.slice(0, 10)} ${isoDateString.slice(11, 19)}`;
        };
        this.timeFormat = (time) => {
            const timeArr = time.split(' ');
            timeArr[0] = timeArr[0].split(':').splice(0, 2).join(':');
            return timeArr.join(' ');
        };
        this.dateFormatter = (date) => {
            return this.dateFormat(date, 1);
        };
        this.daysFormatter = (days) => {
            if (!days) {
                return '-';
            }
            let yr, mt, dy, ret, _yr = 365, _mt = 30;
            if (days > _yr) {
                yr = Math.floor(days / _yr);
            }
            if (days > _mt) {
                mt = Math.floor((days % _yr) / _mt);
            }
            dy = (days % _yr) % _mt;
            ret = `${yr ? this.pluarlizer(yr, 'yr') : ''} ${mt ? this.pluarlizer(mt, 'mth') : ''} ${this.pluarlizer(dy, 'dy')}`;
            return ret;
        };
        this.monthsFormatter = (months) => {
            if (!months) {
                return '-';
            }
            let yr, mt, ret, _yr = 12;
            if (months > _yr) {
                yr = Math.floor(months / _yr);
            }
            mt = months % _yr;
            ret = `${yr ? this.pluarlizer(yr, 'yr') : ''} ${mt ? this.pluarlizer(mt, 'mth') : ''}`;
            return ret;
        };
        this.pluarlizer = (val, txt, plural) => {
            plural = plural || txt + 's';
            return val + (val == 1 ? txt : plural);
        };
        this.decimalToPercentage = (dec) => {
            return dec * 100 + '%';
        };
        this.logForm = (form) => {
            console.log('FORM', form, 'VALUES', form.value);
        };
        this.onlyUnique = (value, index, self) => {
            return self.indexOf(value) === index;
        };
        /**
         * Delete null or undefined fields from an object
         * @param obj Object to delete from
         * @returns
         */
        this.deleteNull = (obj) => {
            for (const key in obj) {
                if (typeof obj[key] == 'object' && obj != null)
                    obj[key] = this.deleteNull(obj[key]);
                if (obj[key] === null || obj[key] === undefined || obj[key] === '')
                    delete obj[key];
            }
            if (!obj || !Object.keys(obj).length)
                return null;
            else
                return obj;
        };
        /**
         * Convert an object into textshowing the field and values
         * @param obj
         * @returns
         */
        this.objectToText = (obj) => {
            return Object.entries(this.deleteNull((0, lodash_1.cloneDeep)(obj)))
                .map((r) => r[0] + ' ' + r[1])
                .join(' / ');
        };
    }
    /**
     * Uses a web worker to perform a task
     * @example new Worker(new URL('./upload-csv.worker', import.meta.url))
     * @param worker Web worker object
     * @param inputData Data being passed into web worker
     * @returns Data gotten from web worker
     */
    useWebWorker(worker, inputData, conditions) {
        return new rxjs_1.Observable((sub) => {
            try {
                if (typeof Worker !== 'undefined') {
                    worker.postMessage(inputData);
                    worker.onmessage = (r) => {
                        // debugger
                        // console.log(e);
                        const data = r.data;
                        if (conditions) {
                            if (conditions.error(data)) {
                                sub.error(data);
                                sub.complete();
                                worker.terminate();
                            }
                            else if (conditions.stop(data)) {
                                sub.next(data);
                                sub.complete();
                                worker.terminate();
                            }
                            else {
                                sub.next(data);
                            }
                        }
                        else {
                            sub.next(data);
                            sub.complete();
                            worker.terminate();
                        }
                    };
                }
                else {
                    throw 'noWorker';
                }
            }
            catch (e) {
                console.log(e);
                sub.error(e);
                sub.complete();
            }
        });
    }
    arrayToMap(arr, keyField, map) {
        const ret = {};
        for (const iterator of arr) {
            ret[iterator[keyField?.toString()]] = map ? map(iterator) : iterator;
        }
        return ret;
    }
    async copyString(str) {
        try {
            const elem = document.createElement('textarea');
            elem.value = str;
            document.body.appendChild(elem);
            elem.select();
            document.execCommand('copy');
            document.body.removeChild(elem);
        }
        catch (e) {
            console.error(e);
        }
    }
    async copyObject(payload) {
        try {
            return this.copyString(JSON.stringify(payload));
        }
        catch (e) {
            console.error(e);
        }
    }
    /**
     * Set an Object's field if the object exists or don't if the object doesn't
     * @param obj Object holding the field
     * @param fieldName Name of the field to set
     * @param val Value to set the field with
     * @returns Returns the object holding the data
     */
    setterWithNull(obj, kvps) {
        if (!obj)
            return null;
        for (const key in kvps) {
            obj[key] = kvps[key];
        }
        return obj;
    }
    /**
     * (Set an Object's field if the object exists or don't if the object doesn't) for multiple entries
     * @param items Array of objects and their modifications
     */
    setterWithNullArray(items) {
        for (const item of items) {
            this.setterWithNull(item.obj, item.kvps);
        }
    }
    removeDuplicate(list, keyField = 'code') {
        if (!list)
            return null;
        list = list.filter((x) => x);
        const ret = [], duplicateMap = {};
        for (const item of list) {
            if (duplicateMap[item[keyField]])
                continue;
            ret.push(item);
            duplicateMap[item[keyField]] = item;
        }
        return ret;
    }
    get genRandomID() {
        return this.generateUUID();
        // return Math.round(Math.random() * 1000000000000000);
    }
    trackByCode(index, item) {
        return item?.code;
    }
    trackByID(index, item) {
        return item?.id;
    }
    /**
     * Converts object to query parameters
     * @param parameters Query object
     * @returns
     */
    objectToQueryParams(parameters) {
        if (!parameters)
            return '';
        return ('?' +
            Object.keys(parameters)
                .filter((key) => parameters[key]?.toString()?.trim() != null)
                .map((key) => `${key}=${parameters[key]?.toString()?.trim()}`)
                .join('&'));
    }
    extractUpload(event) {
        return event.target.files || [];
    }
    isPictureFormat(fileName) {
        if (!fileName)
            return false;
        const PICTURE_FORMATS = ['.jpg', '.jpeg', '.png', '.gif'];
        return PICTURE_FORMATS.some((format) => fileName.endsWith(format));
    }
    objectToArray(obj) {
        const ret = [];
        for (const key in obj) {
            if (Object.prototype.hasOwnProperty.call(obj, key)) {
                ret.push(obj[key]);
            }
        }
        return ret;
    }
    /**
     * Download file from link
     * @param url Link to download file
     * @param filename Name of file including extension
     */
    async downloadFromLink(url, filename) {
        console.log('download link', url);
        // debugger
        let downloadLink = document.createElement('a');
        downloadLink.href = url;
        downloadLink.target = '_blank';
        downloadLink.setAttribute('download', filename);
        document.body.appendChild(downloadLink);
        downloadLink.click();
    }
    exportHTMLToPDF(content, fileName, config) {
        return (0, html2canvas_1.default)(content, { useCORS: true }).then(async (canvas) => {
            let fileWidth = config?.width;
            let fileHeight = config?.height || (canvas.height * fileWidth) / canvas.width;
            // fileHeight = canvas.height
            // fileWidth = canvas.width
            // debugger
            console.log(canvas.width, canvas.height, [fileWidth, fileHeight]);
            const FILEURI = canvas.toDataURL('image/png', 1);
            let PDF = new jspdf_1.default(config?.orientation || 'p', 'px', [fileWidth, fileHeight], true);
            let position = 0;
            PDF.addImage(FILEURI, 'PNG', 0, position, fileWidth, fileHeight);
            return await PDF.save(fileName + '.pdf', {
                returnPromise: true,
            });
        });
    }
    exportHTMLToPDFObject(content, fileName, config) {
        return (0, html2canvas_1.default)(content, { useCORS: true }).then(async (canvas) => {
            let fileWidth = config?.width;
            let fileHeight = config?.height || (canvas.height * fileWidth) / canvas.width;
            // fileHeight = canvas.height
            // fileWidth = canvas.width
            // debugger
            console.log(canvas.width, canvas.height, [fileWidth, fileHeight]);
            const FILEURI = canvas.toDataURL('image/png', 1);
            let PDF = new jspdf_1.default(config?.orientation || 'p', 'px', [fileWidth, fileHeight], true);
            let position = 0;
            PDF.addImage(FILEURI, 'PNG', 0, position, fileWidth, fileHeight);
            return new File([PDF.output('blob')], fileName + '.pdf', {
                type: 'application/pdf',
            });
        });
    }
    arrayToCSV(data, filename, headerMap) {
        const csv = papaparse_1.default.unparse(data, {
            header: false,
            columns: headerMap?.map((x) => x.f?.toString()),
        });
        this.downloader(headerMap.map((fieldName) => fieldName.t).join(',') + `\r\n` + csv, filename?.endsWith('.csv') ? filename : filename + '.csv');
    }
    addDaysToDate(days, date) {
        if (!date)
            return undefined;
        return new Date(new Date(date).getTime() + index_config_1.Config.TimeStampDay * +days)
            .toISOString()
            .split('T')[0];
    }
    get genPhoneNumber() {
        return ('0' +
            new Array(10)
                .fill(1)
                .map((x) => Math.round(Math.random() * 9))
                .join(''));
    }
    generateUUID() {
        let d = new Date().getTime();
        let d2 = (typeof performance !== 'undefined' && performance.now && performance.now() * 1000) || 0; //Time in microseconds since page-load or 0 if unsupported
        return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
            let r = Math.random() * 16; //random number between 0 and 16
            if (d > 0) {
                //Use timestamp until depleted
                r = (d + r) % 16 | 0;
                d = Math.floor(d / 16);
            }
            else {
                //Use microseconds since page-load if supported
                r = (d2 + r) % 16 | 0;
                d2 = Math.floor(d2 / 16);
            }
            return (c === 'x' ? r : (r & 0x3) | 0x8).toString(16);
        });
    }
}
exports.UtilityClass = UtilityClass;
//# sourceMappingURL=utility.class.js.map