"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UtilityClass = void 0;
const la_nest_library_1 = require("@serene-dev/la-nest-library");
const axios_1 = require("axios");
const index_config_1 = require("../configs/index.config");
const environment_service_1 = require("./environment.service");
const promises_1 = require("fs/promises");
const path = require("path");
const rxjs_1 = require("rxjs");
const mail_template_config_1 = require("../configs/mail-template.config");
const authentication_enum_1 = require("../modules/authentication/enums/authentication.enum");
class UtilityClass extends la_nest_library_1.SDKUtilityClass {
    static get relativeUploadPath() {
        return environment_service_1.evt.MULTER_DEST;
    }
    static get absoluteUploadPath() {
        return path.join(this.absolutePath, this.relativeUploadPath);
    }
    static get currentGMTTimestamp() {
        const diff = environment_service_1.evt.timestampDifferenceFromGMT;
        return Date.now() + environment_service_1.evt.timestampDifferenceFromGMT || 0;
    }
    static get currentLocalDateTime() {
        return this.convertDateToLocalDate(Date.now());
    }
    static get currentLocalDate() {
        return this.convertDateTimeToDate(Date.now());
    }
    static yesterdayLocalDate(date = Date.now()) {
        return this.convertDateTimeToDate(new Date(date).getTime() - index_config_1.Config.TimeStampDay);
    }
    static tomorrowLocalDate(date = Date.now()) {
        return this.convertDateTimeToDate(new Date(date).getTime() + index_config_1.Config.TimeStampDay * 3);
    }
    static convertDateToLocalDate(date) {
        return new Date(date).toISOString();
    }
    static convertDateTimeToDate(date) {
        return new Date(date).toISOString().split('T')[0];
    }
    static addDaysToDate(date, days) {
        return this.convertDateTimeToDate(index_config_1.Config.TimeStampDay * days + new Date(date).getTime());
    }
    static calculateTimeZoneDifferenceMilliSec() {
        return (new Date().getTimezoneOffset() + 60) * 60 * 1000;
    }
    static calculateTimeZoneDifferenceMins() {
        return new Date().getTimezoneOffset() + 60;
    }
    static getNextDayDate(nextDay, currentDate = Date.now()) {
        const cd = new Date(currentDate);
        const cdDay = cd.getDay();
        const daysToadd = nextDay - cdDay;
        const cdDate = cd.getDate();
        return _a.convertDateTimeToDate(cd.getDay() == nextDay ? cd : new Date(cd.setDate(cdDate + daysToadd)));
    }
    static dateRange(startDate, endDate, steps = 1) {
        const dateArray = [];
        const currentDate = new Date(startDate);
        while (currentDate <= new Date(endDate)) {
            dateArray.push(new Date(currentDate).toISOString().split('T')[0]);
            currentDate.setUTCDate(currentDate.getUTCDate() + steps);
        }
        return dateArray;
    }
    static getBase64Image(img) {
        const canvas = document.createElement('canvas');
        canvas.width = img.width;
        canvas.height = img.height;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(img, 0, 0);
        const dataURL = canvas.toDataURL('image/png');
        return dataURL.replace(/^data:image\/?[A-z]*;base64,/, '');
    }
    static imageURLToBase64(url) {
        return axios_1.default.get(url).then((r) => {
            return ('data:' +
                r.headers['content-type'] +
                ';base64,' +
                Buffer.from(r.data).toString('base64'));
        });
    }
    static async writeFile(folderPath, fileName, data) {
        const fullPath = `${folderPath}/${fileName}`;
        const _writeFile = () => (0, promises_1.writeFile)(fullPath, data);
        return (0, promises_1.open)(fullPath, 'w')
            .then(async (fh) => {
            await _writeFile();
            fh.close();
        })
            .catch((err) => {
            if (err.errno == -2)
                return (0, promises_1.mkdir)(folderPath).then(() => _writeFile());
            else
                throw err;
        });
    }
    static cacheGenerator(config) {
        const cache = new Map([]);
        const cacheTrimmer$ = new rxjs_1.Subject();
        const cacheTrimmerSub = cacheTrimmer$
            .pipe((0, rxjs_1.debounceTime)(2000))
            .subscribe(() => {
            this.mapCacheTrimmer(cache, config.limit);
            if (config?.verbose)
                console.log('post cache trim ' + (config.cacheName || ''), cache);
        });
        const set = (key, value) => {
            cache.delete(key);
            cache.set(key, value);
            cacheTrimmer$.next();
        };
        return {
            set,
            delete: (key) => {
                cache.delete(key);
            },
            get: (key) => {
                const val = cache.get(key);
                if (val != null)
                    set(key, val);
                return val;
            },
            has: (key) => {
                return cache.has(key);
            },
            trim: () => {
                cacheTrimmer$.next();
            },
            destroy: () => {
                cacheTrimmerSub?.unsubscribe();
            },
        };
    }
    static mapCacheTrimmer(cache, limit) {
        const excess = cache.size - limit;
        if (excess > 0) {
            const arr = Array.from(cache);
            for (let index = 0; index < excess; index++) {
                cache.delete(arr[index][0]);
            }
        }
    }
    static async readFile(filePath, throwErr = false) {
        try {
            return JSON.parse((await (0, promises_1.readFile)(path.join(__dirname, filePath))).toString());
        }
        catch (error) {
            console.error(error);
            if (throwErr)
                _a.throwError({ error });
            else
                return null;
        }
    }
    static async search(repository, query, config) {
        return super.search(repository, query, config);
    }
    static async emailTemplater(header, body) {
        return mail_template_config_1.mailTemplates.general({ header, body });
    }
    static patchSearchWithOrg(query, auth, field = 'orgID') {
        if (auth && auth.userType == authentication_enum_1.EAuthType.provider)
            query[field] = auth.providerId;
        return query;
    }
    static patchSearchWithOwner(query, auth, field = 'creatorId') {
        if (auth && auth.userType == authentication_enum_1.EAuthType.provider)
            query.orgID = auth.providerId;
        else if (auth && auth.userType == authentication_enum_1.EAuthType.public)
            query[field] = auth.id;
        return query;
    }
}
exports.UtilityClass = UtilityClass;
_a = UtilityClass;
UtilityClass.gameSenderName = 'Game Man';
UtilityClass.absolutePath = __dirname;
UtilityClass.convertToDateString = (date) => {
    return _a.datetoJson(date).substr(0, 10);
};
UtilityClass.datetoJson = (date) => {
    const newDate = new Date(date);
    newDate.setMinutes(newDate.getMinutes() + 60);
    return newDate.toJSON();
};
UtilityClass.getGameweekString = (start, end) => {
    return (_a.convertToDateString(start) + '|' + _a.convertToDateString(end));
};
UtilityClass.getGameweekDate = () => {
    const today = new Date();
    const day = today.getDay();
    if (day >= 2) {
        const daysToStartDay = day - 2;
        const date = today.getDate() - daysToStartDay;
        const startDate = new Date();
        startDate.setDate(date);
        const endDate = new Date(startDate);
        endDate.setDate(endDate.getDate() + 6);
        return { startDate, endDate };
    }
    else {
        const daysToEndDay = 1 - day;
        const date = today.getDate() + daysToEndDay;
        const endDate = new Date();
        endDate.setDate(date);
        const startDate = new Date(endDate);
        startDate.setDate(startDate.getDate() - 6);
        return { startDate, endDate };
    }
};
UtilityClass.createQueue = (config) => {
    const queue = [];
    let waitingForTimeoutToCallActioner = false;
    const inProcess = (() => {
        let processes = 0;
        return {
            add: () => ++processes,
            remove: () => --processes,
            hasProcesses: () => processes > 0,
            count: () => processes,
            freeSpace: () => config?.chunkSize - processes,
        };
    })();
    const intervalSeconds = config?.intervalSeconds || (environment_service_1.evt.isDev ? 2 : 60);
    const actioner = () => {
        if (waitingForTimeoutToCallActioner ||
            queue.length == 0 ||
            inProcess.freeSpace() == 0)
            return;
        const slice = queue.splice(0, inProcess.freeSpace());
        console.log('queue state', getQueue());
        slice.forEach(async (s) => {
            inProcess.add();
            try {
                await s.next();
            }
            catch (error) {
                if (error?.['response']?.status == 429 ||
                    error?.['statusCode'] == 429)
                    add(s);
                else {
                    s.onError(error);
                    la_nest_library_1.logger.error({ type: `queue error`, error });
                }
            }
            inProcess.remove();
        });
        debounceActionerCall(intervalSeconds);
    };
    const debounceActionerCall = (seconds) => {
        if (waitingForTimeoutToCallActioner)
            return;
        waitingForTimeoutToCallActioner = true;
        setTimeout(() => {
            waitingForTimeoutToCallActioner = false;
            actioner();
        }, 1000 * seconds);
    };
    const add = (item) => {
        const queueLength = queue.length;
        queue.push(item);
        postAddCheck(queueLength);
    };
    const postAddCheck = (preAddQueueLength) => {
        if (!inProcess.hasProcesses() && preAddQueueLength == 0) {
            debounceActionerCall(1);
        }
    };
    const getQueue = () => ({
        currentTime: new Date(),
        freeSpace: inProcess.freeSpace(),
        inProcess: inProcess.hasProcesses(),
        inProcessCount: inProcess.count(),
        intervalInSeconds: intervalSeconds,
        itemsWaitingInQueue: queue.length,
        waitingForTimeoutToCallActioner: waitingForTimeoutToCallActioner,
    });
    return {
        getQueue,
        clear: () => queue.splice(0),
        addBatch: (items) => {
            const queueLength = queue.length;
            queue.push(...items);
            postAddCheck(queueLength);
        },
        add,
    };
};
UtilityClass.isAdmin = (auth) => auth.userType == authentication_enum_1.EAuthType.admin;
UtilityClass.isProvider = (auth) => auth.userType == authentication_enum_1.EAuthType.provider;
UtilityClass.isPublic = (auth) => auth.userType == authentication_enum_1.EAuthType.public;
la_nest_library_1.SDKUtilityClass.emailTemplater = UtilityClass.emailTemplater;
la_nest_library_1.SDKUtilityClass.swaggerUILink = 'http://localhost:8100/api-docs';
//# sourceMappingURL=utility.service.js.map